﻿using System;
using System.Collections.Generic;
using System.Linq;
using VA.TMP.DataModel;
using VA.TMP.Integration.VIMT.Shared;
using VA.TMP.Integration.VIMT.VirtualMeetingRoom.StateObject;
using VRM.Integration.Servicebus.Core;

namespace VA.TMP.Integration.VIMT.VirtualMeetingRoom.PipelineSteps.OnDemand
{
    /// <summary>
    /// Set the Meeting Room Name Step. 
    /// </summary>
    public class SetMeetingRoomNameStep : FilterBase<VmrOnDemandCreateStateObject>
    {
        /// <summary>
        /// Execute the step.
        /// </summary>
        /// <param name="state">State object.</param>
        public override void Execute(VmrOnDemandCreateStateObject state)
        {
            using (var context = new Xrm(state.OrganizationServiceProxy))
            {
                var videoOnDemand = context.cvt_vodSet.First(x => x.Id == state.VideoOnDemandId);

                if (videoOnDemand.cvt_starttime == null) throw new Exception("The Scheduled Start of the Video On Demand is null. Unable to continue processing.");

                // Get the date one hundred days prior to the Start of the Video On Demand.
                var dateHundredDaysAgo = videoOnDemand.cvt_starttime.Value.Subtract(new TimeSpan(100, 0, 0, 0));

                //Todo: Question - Do we need to check the ramdon number in both Service appointment and Video On Demand entities?
                // In order to generate the Virtual Meeting Room name we have to have a random 7 digit number that does not already exist 100 days prior to the appointment.
                var allVodsScheduledForMonth = context.cvt_vodSet
                    .Where(x => x.Id != state.VideoOnDemandId && x.cvt_meetingroomname != null && x.cvt_starttime >= dateHundredDaysAgo && x.cvt_starttime <= videoOnDemand.cvt_starttime.Value)
                    .ToList()
                    .Where(x => x.cvt_meetingroomname.Length == (state.VirtualMeetingRoomPrefix.Length + state.VirtualMeetingRoomDigitLength))
                    .Select(x => x.cvt_meetingroomname.Substring(3, state.VirtualMeetingRoomDigitLength))
                    .ToDictionary(k => k, v => string.Empty);

                // If there are previous Meeting Rooms within the same month then call the helper method to find the next unique one; otherwise just generate it.
                state.MeetingRoomName = string.Format("{0}{1}", state.VirtualMeetingRoomPrefix, allVodsScheduledForMonth.Count > 0
                    ? GetNextVirtualMeetingRoomSuffix(allVodsScheduledForMonth, state.VirtualMeetingRoomDigitLength)
                    : RandomDigits.GetRandomDigitString(state.VirtualMeetingRoomDigitLength));
            }
        }

        /// <summary>
        /// Determine if a random string exists in the collection. If so then generate until there is a unique one.
        /// </summary>
        /// <param name="allVodsScheduledForMonth">Dictionary of Virtual Meeting Room Suffixes.</param>
        /// <param name="virtualMeetingRoomDigitLength">Length of Virtual Meeting Room prefix.</param>
        /// <returns>Random five digit numeric string that is not used for the current month.</returns>
        private static string GetNextVirtualMeetingRoomSuffix(IReadOnlyDictionary<string, string> allVodsScheduledForMonth, int virtualMeetingRoomDigitLength)
        {
            while (true)
            {
                var randomDigit = RandomDigits.GetRandomDigitString(virtualMeetingRoomDigitLength);

                if (allVodsScheduledForMonth.ContainsKey(randomDigit)) continue;

                return randomDigit;
            }
        }
    }
}